require('dotenv').config();
const express = require('express');
const session = require('express-session');
const passport = require('passport');
const LocalStrategy = require('passport-local').Strategy;
const bcrypt = require('bcryptjs');
const path = require('path');
const { sequelize } = require('./config/database');
const { User, Crime, Case } = require('./models');
const { Op } = require('sequelize');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, 'public')));

// View engine setup
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Session configuration
app.use(session({
    secret: process.env.SESSION_SECRET || 'your-secret-key', // Change this in production
    resave: false,
    saveUninitialized: false,
    cookie: { secure: process.env.NODE_ENV === 'production' }
}));

// Flash messages middleware
const flash = require('connect-flash');
app.use(flash());

// Set up local variables for all views
app.use((req, res, next) => {
    res.locals.success_msg = req.flash('success_msg');
    res.locals.error_msg = req.flash('error_msg');
    res.locals.error = req.flash('error');
    res.locals.user = req.user || null;
    next();
});

// Passport configuration
app.use(passport.initialize());
app.use(passport.session());

passport.use(new LocalStrategy(
    async (username, password, done) => {
        try {
            // Trim and ensure consistent case for username
            const cleanUsername = username ? username.trim() : '';
            
            // Find user with case-insensitive username match
            const user = await User.findOne({ 
                where: sequelize.where(
                    sequelize.fn('LOWER', sequelize.col('username')), 
                    'LIKE', 
                    `%${cleanUsername.toLowerCase()}%`
                )
            });
            
            console.log('Looking for user with username:', `'${cleanUsername}'`);
            if (!user) {
                return done(null, false, { message: 'Incorrect username.' });
            }
            
            const isValidPassword = await bcrypt.compare(password, user.password_hash);
            if (!isValidPassword) {
                return done(null, false, { message: 'Incorrect password.' });
            }
            
            return done(null, user);
        } catch (error) {
            return done(error);
        }
    }
));

passport.serializeUser((user, done) => {
    done(null, user.id);
});

passport.deserializeUser(async (id, done) => {
    try {
        const user = await User.findByPk(id, {
            attributes: { include: ['is_admin'] }  // Explicitly include is_admin
        });
        if (!user) {
            return done(null, false);
        }
        done(null, user.get({ plain: true }));
    } catch (error) {
        console.error('Error deserializing user:', error);
        done(error);
    }
});

// Routes
const authRoutes = require('./routes/auth');
const indexRoutes = require('./routes');
const adminRoutes = require('./routes/admin');
const casesRoutes = require('./routes/cases');

// Redirect /login to /auth/login for convenience
app.get('/login', (req, res) => {
    res.redirect('/auth/login');
});
// Logout route (for compatibility with old links)
app.get('/logout', (req, res, next) => {
    req.logout(function(err) {
        if (err) { 
            console.error('Error during logout:', err);
            return next(err); 
        }
        req.flash('success_msg', 'You have been successfully logged out.');
        res.redirect('/auth/login');
    });
});



app.get('/debug/user', (req, res) => {
    if (!req.user) {
        return res.status(200).json({
            isAuthenticated: false,
            message: 'No user is currently logged in',
            session: req.session,
            user: null
        });
    }
    
    // Get the raw user data without trying to call .get()
    const userData = {
        id: req.user.id,
        username: req.user.username,
        is_admin: req.user.is_admin,
        rank: req.user.rank,
        last_login: req.user.last_login,
        // Include any other user fields you need
        rawUser: req.user  // This is already a plain object
    };
    
    res.status(200).json({
        isAuthenticated: true,
        user: userData,
        session: req.session,
        sessionUser: req.session.passport ? req.session.passport.user : null
    });
});

app.use('/auth', authRoutes);
app.use('/admin', adminRoutes);
app.use('/cases', casesRoutes);
app.use('/', indexRoutes);
app.use('/api/crimes', require('./routes/api/crimes'));
app.use('/api/cases', require('./routes/api/cases'));

// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).send('Something broke!');
});

// Sync database and start server
const init = async () => {
    try {
        await sequelize.authenticate();
        console.log('Database connection has been established successfully.');
        
        // Sync all models
        await sequelize.sync({ force: false });
        console.log('Database synced');
        
        // Create default admin user if it doesn't exist
        await createDefaultAdmin();
        
        // Create some default crimes if none exist
        const crimeCount = await Crime.count();
        if (crimeCount === 0) {
            await Crime.bulkCreate([
                {
                    name: 'Theft Under $500',
                    description: 'Petty theft of items under $500',
                    fine: 500,
                    jail_minutes: 30,
                    is_felony: false,
                    is_active: true
                },
                {
                    name: 'Theft $500-$1000',
                    description: 'Theft of items between $500 and $1000',
                    fine: 1000,
                    jail_minutes: 180,
                    is_felony: false,
                    is_active: true
                },
                {
                    name: 'Assault',
                    description: 'Simple assault without serious injury',
                    fine: 2000,
                    jail_minutes: 1440, // 1 day
                    is_felony: true,
                    is_active: true
                },
                {
                    name: 'Burglary',
                    description: 'Unlawful entry into a building with intent to commit a crime',
                    fine: 5000,
                    jail_minutes: 10080, // 1 week
                    is_felony: true,
                    is_active: true
                },
                {
                    name: 'Drug Possession',
                    description: 'Possession of controlled substances',
                    fine: 1500,
                    jail_minutes: 720, // 12 hours
                    is_felony: true,
                    is_active: true
                }
            ]);
            console.log('Added default crimes');
        }
        
        app.listen(PORT, () => {
            console.log(`Server is running on http://localhost:${PORT}`);
            console.log(`Admin panel: http://localhost:${PORT}/admin`);
            console.log(`Crime calculator: http://localhost:${PORT}/calculator`);
        });
    } catch (error) {
        console.error('Unable to connect to the database:', error);
    }
};

const createDefaultAdmin = async () => {
    try {
        // First, check if any admin exists
        const adminExists = await User.findOne({ 
            where: { 
                [Op.or]: [
                    { username: 'admin' },
                    { is_admin: true }
                ]
            } 
        });

        if (!adminExists) {
            // Create admin with all necessary fields
            const admin = await User.create({
                username: 'admin',
                password_hash: 'admin123', // Will be hashed by the model's beforeCreate hook
                email: 'admin@policedb.com',
                first_name: 'System',
                last_name: 'Administrator',
                rank: 'Chief',
                badge_number: 'ADMIN001',
                department: 'Administration',
                is_admin: true,
                is_active: true,
                role: 'admin',
                last_login: new Date(),
                created_at: new Date(),
                updated_at: new Date()
            });
            
            console.log('========================================');
            console.log('ADMIN CREDENTIALS');
            console.log('========================================');
            console.log('Username: admin');
            console.log('Password: admin123');
            console.log('========================================');
            console.log('IMPORTANT: Change this password after first login!');
            console.log('========================================');
            
            // Verify the password can be checked
            const isValid = await admin.validPassword('admin123');
            console.log('Password verification test:', isValid ? 'SUCCESS' : 'FAILED');
            
            return admin;
        } else {
            console.log('Admin user already exists');
            return adminExists;
        }
    } catch (error) {
        console.error('Error creating admin user:', error);
        throw error; // Re-throw to prevent server start if admin creation fails
    }
};

// Initialize the application
init();
