const { Crime, User } = require('../models');

// Get all crimes for admin
const getCrimes = async (req, res) => {
    try {
        const crimes = await Crime.findAll({
            order: [['name', 'ASC']]
        });
        res.render('admin/crimes', {
            title: 'Manage Crimes',
            crimes,
            user: req.user
        });
    } catch (error) {
        console.error('Error fetching crimes:', error);
        req.flash('error', 'Error loading crimes');
        res.redirect('/admin');
    }
};

// Show form to add new crime
const showAddCrimeForm = (req, res) => {
    res.render('admin/crimes-form', {
        title: 'Add New Crime',
        crime: null,
        user: req.user,
        formAction: '/admin/crimes',
        formMethod: 'POST',
        submitButtonText: 'Add Crime'
    });
};

// Add new crime
const addCrime = async (req, res) => {
    try {
        const { name, description, fine, jail_minutes, is_felony, is_active } = req.body;
        
        await Crime.create({
            name,
            description: description || null,
            fine: parseInt(fine) || 0,
            jail_minutes: parseInt(jail_minutes) || 0,
            is_felony: is_felony === 'on',
            is_active: is_active === 'on'
        });
        
        req.flash('success', 'Crime added successfully');
        res.redirect('/admin/crimes');
    } catch (error) {
        console.error('Error adding crime:', error);
        req.flash('error', 'Error adding crime');
        res.redirect('/admin/crimes/new');
    }
};

// Show form to edit crime
const showEditCrimeForm = async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            req.flash('error', 'Crime not found');
            return res.redirect('/admin/crimes');
        }
        
        res.render('admin/crimes-form', {
            title: 'Edit Crime',
            crime,
            user: req.user,
            formAction: `/admin/crimes/${crime.id}`, 
            formMethod: 'POST',
            submitButtonText: 'Update Crime',
            _method: 'PUT'
        });
    } catch (error) {
        console.error('Error fetching crime:', error);
        req.flash('error', 'Error loading crime');
        res.redirect('/admin/crimes');
    }
};

// Update crime
const updateCrime = async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            req.flash('error', 'Crime not found');
            return res.redirect('/admin/crimes');
        }
        
        const { name, description, fine, jail_minutes, is_felony, is_active } = req.body;
        
        await crime.update({
            name,
            description: description || null,
            fine: parseInt(fine) || 0,
            jail_minutes: parseInt(jail_minutes) || 0,
            is_felony: is_felony === 'on',
            is_active: is_active === 'on'
        });
        
        req.flash('success', 'Crime updated successfully');
        res.redirect('/admin/crimes');
    } catch (error) {
        console.error('Error updating crime:', error);
        req.flash('error', 'Error updating crime');
        res.redirect(`/admin/crimes/${req.params.id}/edit`);
    }
};

// Delete crime
const deleteCrime = async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            req.flash('error', 'Crime not found');
            return res.redirect('/admin/crimes');
        }
        
        await crime.destroy();
        req.flash('success', 'Crime deleted successfully');
        res.redirect('/admin/crimes');
    } catch (error) {
        console.error('Error deleting crime:', error);
        req.flash('error', 'Error deleting crime');
        res.redirect('/admin/crimes');
    }
};

// Toggle crime status
const toggleCrimeStatus = async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            return res.status(404).json({ success: false, message: 'Crime not found' });
        }
        
        crime.is_active = !crime.is_active;
        await crime.save();
        
        res.json({ 
            success: true, 
            is_active: crime.is_active 
        });
    } catch (error) {
        console.error('Error toggling crime status:', error);
        res.status(500).json({ success: false, message: 'Error toggling crime status' });
    }
};

// Get all users for admin
const getUsers = async (req, res) => {
    try {
        const users = await User.findAll({
            attributes: ['id', 'username', 'rank', 'is_admin', 'last_login', 'created_at'],
            order: [['username', 'ASC']]
        });
        
        res.render('admin/users', {
            title: 'Manage Users',
            users,
            user: req.user,
            req: req // Pass req for checking current user
        });
    } catch (error) {
        console.error('Error fetching users:', error);
        req.flash('error', 'Error loading users');
        res.redirect('/admin');
    }
};

// Update user
const updateUser = async (req, res) => {
    try {
        const { id, username, rank, is_admin, password } = req.body;
        
        const user = await User.findByPk(id);
        if (!user) {
            req.flash('error', 'User not found');
            return res.redirect('/admin/users');
        }
        
        // Don't let users demote themselves from admin
        if (user.id === req.user.id && is_admin === 'false') {
            req.flash('error', 'You cannot remove your own admin status');
            return res.redirect('/admin/users');
        }
        
        // Update user data
        const updateData = {
            username,
            rank: is_admin === 'true' ? 'admin' : rank,
            is_admin: is_admin === 'true'
        };
        
        // Only update password if provided
        if (password && password.trim() !== '') {
            updateData.password_hash = password; // Will be hashed by the model hook
        }
        
        await user.update(updateData);
        
        req.flash('success', 'User updated successfully');
        res.redirect('/admin/users');
    } catch (error) {
        console.error('Error updating user:', error);
        req.flash('error', 'Error updating user');
        res.redirect('/admin/users');
    }
};

// Delete user
const deleteUser = async (req, res) => {
    try {
        const userId = req.params.id;
        
        // Prevent deleting yourself
        if (parseInt(userId) === req.user.id) {
            req.flash('error', 'You cannot delete your own account');
            return res.redirect('/admin/users');
        }
        
        const user = await User.findByPk(userId);
        if (!user) {
            req.flash('error', 'User not found');
            return res.redirect('/admin/users');
        }
        
        await user.destroy();
        
        req.flash('success', 'User deleted successfully');
        res.redirect('/admin/users');
    } catch (error) {
        console.error('Error deleting user:', error);
        req.flash('error', 'Error deleting user');
        res.redirect('/admin/users');
    }
};

module.exports = {
    getCrimes,
    showAddCrimeForm,
    addCrime,
    showEditCrimeForm,
    updateCrime,
    deleteCrime,
    toggleCrimeStatus,
    getUsers,
    updateUser,
    deleteUser
};
