const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Case = sequelize.define('Case', {
    id: {
        type: DataTypes.INTEGER,
        primaryKey: true,
        autoIncrement: true
    },
    suspect_name: {
        type: DataTypes.STRING(100),
        allowNull: false
    },
    suspect_identifier: {
        type: DataTypes.STRING(50), // Could be citizen ID, license, etc.
        allowNull: true
    },
    total_fine: {
        type: DataTypes.INTEGER,
        allowNull: false,
        defaultValue: 0
    },
    total_jail_minutes: {
        type: DataTypes.INTEGER,
        allowNull: false,
        defaultValue: 0
    },
    notes: {
        type: DataTypes.TEXT,
        allowNull: true
    },
    status: {
        type: DataTypes.ENUM('pending', 'processed', 'dismissed'),
        defaultValue: 'pending'
    },
    officer_id: {
        type: DataTypes.INTEGER,
        allowNull: true,
        references: {
            model: 'police_officers',
            key: 'id',
        },
        onUpdate: 'CASCADE',
        onDelete: 'SET NULL'
    }
}, {
    tableName: 'police_cases',
    timestamps: true,
    underscored: true
});

// Instance method to add crimes to a case
Case.prototype.addCrimes = async function(crimeIds) {
    try {
        await this.setCrimes(crimeIds);
        
        // Calculate total fine and jail time
        const crimes = await this.getCrimes();
        const totals = crimes.reduce((acc, crime) => ({
            fine: acc.fine + (crime.fine || 0),
            jail: acc.jail + (crime.jail_minutes || 0)
        }), { fine: 0, jail: 0 });
        
        this.total_fine = totals.fine;
        this.total_jail_minutes = totals.jail;
        
        await this.save();
        return this;
    } catch (error) {
        console.error('Error adding crimes to case:', error);
        throw error;
    }
};

module.exports = Case;
