const express = require('express');
const router = express.Router();
const { Case, Crime } = require('../../models');
const { isAuthenticated } = require('../../middleware/auth');

// Create a new case
router.post('/', isAuthenticated, async (req, res) => {
    try {
        const { suspect_name, suspect_identifier, crime_ids, notes } = req.body;
        
        // Create the case
        const newCase = await Case.create({
            suspect_name,
            suspect_identifier,
            notes,
            officer_id: req.user.id,
            status: 'pending',
            total_fine: 0,
            total_jail_minutes: 0
        });
        
        // Add crimes to the case and calculate totals
        if (crime_ids && crime_ids.length > 0) {
            const crimes = await Crime.findAll({
                where: { id: crime_ids }
            });
            
            await newCase.setCrimes(crimes);
            
            // Calculate totals
            const totals = crimes.reduce((acc, crime) => ({
                fine: acc.fine + (crime.fine || 0),
                jail: acc.jail + (crime.jail_minutes || 0)
            }), { fine: 0, jail: 0 });
            
            // Update case with totals
            await newCase.update({
                total_fine: totals.fine,
                total_jail_minutes: totals.jail
            });
        }
        
        // Return the full case with related data
        const result = await Case.findByPk(newCase.id, {
            include: [
                { model: Crime, through: { attributes: [] } },
                { model: require('../../models').User, attributes: ['username', 'rank'] }
            ]
        });
        
        res.status(201).json(result);
    } catch (error) {
        console.error('Error creating case:', error);
        res.status(500).json({ error: 'Error creating case' });
    }
});

// Get case by ID
router.get('/:id', isAuthenticated, async (req, res) => {
    try {
        const caseItem = await Case.findByPk(req.params.id, {
            include: [
                { model: Crime, through: { attributes: [] } },
                { model: require('../../models').User, attributes: ['username', 'rank'] }
            ]
        });
        
        if (!caseItem) {
            return res.status(404).json({ error: 'Case not found' });
        }
        
        res.json(caseItem);
    } catch (error) {
        console.error('Error fetching case:', error);
        res.status(500).json({ error: 'Error fetching case' });
    }
});

// Update case status
router.patch('/:id/status', isAuthenticated, async (req, res) => {
    try {
        const { status } = req.body;
        const caseItem = await Case.findByPk(req.params.id);
        
        if (!caseItem) {
            return res.status(404).json({ error: 'Case not found' });
        }
        
        // Verify the user is the owner or an admin
        if (caseItem.officer_id !== req.user.id && !req.user.is_admin) {
            return res.status(403).json({ error: 'Not authorized' });
        }
        
        await caseItem.update({ status });
        res.json(caseItem);
    } catch (error) {
        console.error('Error updating case status:', error);
        res.status(500).json({ error: 'Error updating case status' });
    }
});

// Delete case (soft delete)
router.delete('/:id', isAuthenticated, async (req, res) => {
    try {
        const caseItem = await Case.findByPk(req.params.id);
        
        if (!caseItem) {
            return res.status(404).json({ error: 'Case not found' });
        }
        
        // Only allow admin or the officer who created the case to delete it
        if (caseItem.officer_id !== req.user.id && !req.user.is_admin) {
            return res.status(403).json({ error: 'Not authorized' });
        }
        
        // Soft delete
        await caseItem.update({ status: 'deleted' });
        res.json({ message: 'Case marked as deleted' });
    } catch (error) {
        console.error('Error deleting case:', error);
        res.status(500).json({ error: 'Error deleting case' });
    }
});

module.exports = router;
