const express = require('express');
const router = express.Router();
const { Crime } = require('../../models');
const { isAuthenticated, isAdmin } = require('../../middleware/auth');

// Get all crimes (public)
router.get('/', async (req, res) => {
    try {
        const crimes = await Crime.findAll({
            where: { is_active: true },
            order: [['name', 'ASC']]
        });
        res.json(crimes);
    } catch (error) {
        console.error('Error fetching crimes:', error);
        res.status(500).json({ error: 'Error fetching crimes' });
    }
});

// Create new crime (admin only)
router.post('/', isAuthenticated, isAdmin, async (req, res) => {
    try {
        const crime = await Crime.create(req.body);
        res.status(201).json(crime);
    } catch (error) {
        console.error('Error creating crime:', error);
        res.status(500).json({ error: 'Error creating crime' });
    }
});

// Update crime (admin only)
router.put('/:id', isAuthenticated, isAdmin, async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            return res.status(404).json({ error: 'Crime not found' });
        }
        
        await crime.update(req.body);
        res.json(crime);
    } catch (error) {
        console.error('Error updating crime:', error);
        res.status(500).json({ error: 'Error updating crime' });
    }
});

// Delete crime (admin only, soft delete)
router.delete('/:id', isAuthenticated, isAdmin, async (req, res) => {
    try {
        const crime = await Crime.findByPk(req.params.id);
        if (!crime) {
            return res.status(404).json({ error: 'Crime not found' });
        }
        
        // Soft delete
        await crime.update({ is_active: false });
        res.json({ message: 'Crime deactivated successfully' });
    } catch (error) {
        console.error('Error deleting crime:', error);
        res.status(500).json({ error: 'Error deleting crime' });
    }
});

module.exports = router;
