const express = require('express');
const router = express.Router();
const passport = require('passport');
const { User } = require('../models');
const { isAuthenticated, isAdmin } = require('../middleware/auth');

// Login page
router.get('/login', (req, res) => {
    if (req.isAuthenticated()) {
        return res.redirect('/dashboard');
    }
    const error = req.flash('error');
    const error_msg = req.flash('error_msg');
    const success_msg = req.flash('success_msg');
    
    res.render('auth/login', { 
        title: 'Login',
        error: error.length > 0 ? error : null,
        error_msg: error_msg.length > 0 ? error_msg : null,
        success_msg: success_msg.length > 0 ? success_msg : null,
        username: ''
    });
});

// Logout route
router.get('/logout', (req, res) => {
    req.logout(function(err) {
        if (err) { 
            console.error('Error during logout:', err);
            return next(err); 
        }
        req.flash('success_msg', 'You have been successfully logged out.');
        res.redirect('/auth/login');
    });
});

// Handle login
router.post('/login', (req, res, next) => {
    // Trim whitespace from username
    if (req.body.username) {
        req.body.username = req.body.username.trim();
    }
    console.log('Login attempt for user:', `'${req.body.username}'`);
    
    passport.authenticate('local', (err, user, info) => {
        console.log('Passport authenticate callback:', { err, user: user ? 'User found' : 'No user', info });
        
        if (err) {
            console.error('Authentication error:', err);
            req.flash('error', err.message || 'An error occurred during login');
            return res.redirect('/login');
        }
        
        if (!user) {
            console.log('Authentication failed:', info.message);
            req.flash('error', info.message || 'Invalid username or password');
            return res.render('auth/login', {
                title: 'Login',
                error: [info.message || 'Invalid username or password'],
                username: req.body.username || ''
            });
        }
        
        req.logIn(user, async (err) => {
            if (err) {
                console.error('Session error during login:', err);
                req.flash('error', 'Error during login');
                return next(err);
            }
            
            try {
                // Update last login
                user.last_login = new Date();
                await user.save();
                console.log('User logged in successfully:', user.username);
                
                req.flash('success', 'Successfully logged in');
                return res.redirect('/dashboard');
            } catch (saveErr) {
                console.error('Error updating last login:', saveErr);
                req.flash('error', 'Error updating user session');
                return res.redirect('/login');
            }
        });
    })(req, res, next);
});

// Logout
router.get('/logout', (req, res) => {
    req.logout();
    req.flash('success', 'Successfully logged out');
    res.redirect('/login');
});

// Registration page (admin only)
router.get('/register', isAuthenticated, isAdmin, (req, res) => {
    res.render('auth/register', { title: 'Register New Officer' });
});

// Handle registration (admin only)
router.post('/register', isAuthenticated, isAdmin, async (req, res) => {
    try {
        const { username, password, rank } = req.body;
        
        // Check if username already exists
        const existingUser = await User.findOne({ where: { username } });
        if (existingUser) {
            req.flash('error', 'Username already exists');
            return res.redirect('/auth/register');
        }
        
        // Create new user (password will be hashed by the model hook)
        await User.create({
            username,
            password_hash: password, // Will be hashed by the model hook
            rank: rank || 'officer',
            is_admin: req.body.is_admin === 'on'
        });
        
        req.flash('success', 'Officer registered successfully');
        res.redirect('/dashboard');
    } catch (error) {
        console.error('Registration error:', error);
        req.flash('error', 'Error registering new officer');
        res.redirect('/auth/register');
    }
});

module.exports = router;
