const express = require('express');
const router = express.Router();
const { isAuthenticated, isAdmin, setLocals } = require('../middleware/auth');
const { Crime, Case, User } = require('../models');
const { Op } = require('sequelize');

// Apply setLocals middleware to all routes
router.use(setLocals);

// Home page (redirects to login or dashboard)
router.get('/', (req, res) => {
    if (req.isAuthenticated()) {
        return res.redirect('/dashboard');
    }
    res.redirect('/login');
});

// Dashboard - No JavaScript version
router.get('/dashboard', isAuthenticated, async (req, res) => {
    try {
        const { search: searchTerm } = req.query;
        
        // Get all active crimes (filter by search term if provided)
        const whereClause = { is_active: true };
        if (searchTerm) {
            whereClause[Op.or] = [
                { name: { [Op.like]: `%${searchTerm}%` } },
                { description: { [Op.like]: `%${searchTerm}%` } }
            ];
        }
        
        // Get crimes from database using the model from app.js
        const crimes = await Crime.findAll({
            where: whereClause,
            attributes: ['id', 'name', 'description', 'fine', 'jail_minutes', 'is_felony'],
            order: [['name', 'ASC']],
            raw: true
        });
        
        // Get selected crimes from session or initialize empty array
        const selectedCrimes = req.session.selectedCrimes || [];
        
        res.render('dashboard/nojs-dashboard', {
            title: 'Crime Calculator',
            user: req.user,
            crimes,
            selectedCrimes,
            searchTerm: searchTerm || ''
        });
    } catch (error) {
        console.error('Dashboard error:', error);
        res.status(500).send('Error loading dashboard');
    }
});

// Add crime to selection
router.post('/dashboard/add-crime', isAuthenticated, async (req, res) => {
    try {
        const { crimeId } = req.body;
        
        // Get the crime details
        const crime = await Crime.findByPk(crimeId, {
            attributes: ['id', 'name', 'fine', 'jail_minutes', 'is_felony']
        });
        
        if (!crime) {
            req.flash('error', 'Crime not found');
            return res.redirect('/dashboard');
        }
        
        // Initialize selected crimes array in session if it doesn't exist
        if (!req.session.selectedCrimes) {
            req.session.selectedCrimes = [];
        }
        
        // Check if crime is already selected
        const existingIndex = req.session.selectedCrimes.findIndex(c => c.id === crime.id);
        
        if (existingIndex >= 0) {
            // Increment quantity if already selected
            req.session.selectedCrimes[existingIndex].quantity += 1;
        } else {
            // Add new crime to selection
            req.session.selectedCrimes.push({
                id: crime.id,
                name: crime.name,
                fine: parseFloat(crime.fine),
                jail_minutes: crime.jail_minutes,
                is_felony: crime.is_felony,
                quantity: 1
            });
        }
        
        res.redirect('/dashboard');
    } catch (error) {
        console.error('Error adding crime:', error);
        req.flash('error', 'Error adding crime');
        res.redirect('/dashboard');
    }
});

// Update crime quantity
router.post('/dashboard/update-quantity', isAuthenticated, (req, res) => {
    try {
        const { crimeIndex, newQuantity } = req.body;
        
        if (!req.session.selectedCrimes || !req.session.selectedCrimes[crimeIndex]) {
            req.flash('error', 'Invalid crime selection');
            return res.redirect('/dashboard');
        }
        
        const quantity = parseInt(newQuantity, 10);
        if (isNaN(quantity) || quantity < 1) {
            req.flash('error', 'Invalid quantity');
            return res.redirect('/dashboard');
        }
        
        req.session.selectedCrimes[crimeIndex].quantity = quantity;
        res.redirect('/dashboard');
    } catch (error) {
        console.error('Error updating quantity:', error);
        req.flash('error', 'Error updating quantity');
        res.redirect('/dashboard');
    }
});

// Remove crime from selection
router.post('/dashboard/remove-crime', isAuthenticated, (req, res) => {
    try {
        const { crimeIndex } = req.body;
        
        if (!req.session.selectedCrimes || !req.session.selectedCrimes[crimeIndex]) {
            req.flash('error', 'Invalid crime selection');
            return res.redirect('/dashboard');
        }
        
        // Remove the crime from the array
        req.session.selectedCrimes.splice(crimeIndex, 1);
        
        res.redirect('/dashboard');
    } catch (error) {
        console.error('Error removing crime:', error);
        req.flash('error', 'Error removing crime');
        res.redirect('/dashboard');
    }
});

// Clear all selected crimes
router.post('/dashboard/clear-crimes', isAuthenticated, (req, res) => {
    try {
        req.session.selectedCrimes = [];
        res.redirect('/dashboard');
    } catch (error) {
        console.error('Error clearing crimes:', error);
        req.flash('error', 'Error clearing crimes');
        res.redirect('/dashboard');
    }
});

// Calculator page
router.get('/calculator', isAuthenticated, (req, res) => {
    try {
        res.render('calculator/index', {
            title: 'Crime Calculator',
            user: req.user
        });
    } catch (error) {
        console.error('Error loading calculator:', error);
        req.flash('error', 'Error loading calculator');
        res.redirect('/dashboard');
    }
});

// API: Get all crimes for calculator
router.get('/api/crimes', isAuthenticated, async (req, res) => {
    try {
        const crimes = await Crime.findAll({
            where: { is_active: true },
            attributes: ['id', 'name', 'description', 'fine', 'jail_minutes', 'is_felony'],
            order: [['name', 'ASC']]
        });
        
        // Format the crimes data for the frontend
        const formattedCrimes = crimes.map(crime => ({
            id: crime.id,
            name: crime.name,
            description: crime.description,
            fine: crime.fine,
            jailTime: crime.jail_minutes,
            category: crime.is_felony ? 'Felony' : 'Misdemeanor'
        }));
        
        res.json(formattedCrimes);
    } catch (error) {
        console.error('Error fetching crimes:', error);
        res.status(500).json({ error: 'Failed to fetch crimes' });
    }
});

// Cases list
router.get('/cases', isAuthenticated, async (req, res) => {
    try {
        const cases = await Case.findAll({
            order: [['created_at', 'DESC']],
            include: [
                { model: Crime, through: { attributes: [] } },
                { model: User, attributes: ['username', 'rank'] }
            ]
        });
        
        res.render('cases/index', {
            title: 'All Cases',
            cases
        });
    } catch (error) {
        console.error('Error fetching cases:', error);
        req.flash('error', 'Error loading cases');
        res.redirect('/dashboard');
    }
});

// View single case
router.get('/cases/:id', isAuthenticated, async (req, res) => {
    try {
        const caseItem = await Case.findByPk(req.params.id, {
            include: [
                { model: Crime, through: { attributes: [] } },
                { model: User, attributes: ['username', 'rank'] }
            ]
        });

        if (!caseItem) {
            req.flash('error', 'Case not found');
            return res.redirect('/cases');
        }

        res.render('cases/view', {
            title: `Case #${caseItem.id}`,
            case: caseItem
        });
    } catch (error) {
        console.error('Error fetching case:', error);
        req.flash('error', 'Error loading case');
        res.redirect('/cases');
    }
});

// Crime management (admin only)
router.get('/admin/crimes', isAuthenticated, isAdmin, async (req, res) => {
    try {
        const crimes = await Crime.findAll({
            order: [['name', 'ASC']]
        });
        
        res.render('admin/crimes', {
            title: 'Manage Crimes',
            crimes
        });
    } catch (error) {
        console.error('Error fetching crimes:', error);
        req.flash('error', 'Error loading crimes');
        res.redirect('/dashboard');
    }
});

module.exports = router;
